PRO SeAFFluX_v2

  ;###############################################################################################
  ;# Program title: SeAFFluX version 1.1 (Self-Absorption Fitting for Fluorescent X-rays)        #
  ;# Author: Ryan Trevorah (inspired by previous work by Tauhid Islam and Nick Rae)              #
  ;# Date created: July-Aug 2016                                                                 #
  ;# Latest update: Dec 2018                                                                     #
  ;# Program Application: Fits and corrects for the self-absorption systematic found in          #
  ;#                      fluorescent high-energy X-ray measurements according to theory         #
  ;#                      published by Trevorah, Chantler and Schalken (IUCrJ, 2018).            #                        
  ;###############################################################################################

  ; Reading in data.
  path = cgsourcedir()
  
  f1 = path + 'Nickel_td_3scans.txt' ; data file with formatted 15mM ipro QXAFS data.

  header = ''
  openr, unit1, f1, /get_lun
  readf, unit1, header

 ; Defining structure of data columns
  stc =  {ENERGY:0.d, MONOCHROM:0.d,ENCODER_MONO:0.d,TEMPC:0.d,EPOCH:0.d,SECONDS:0.d,DETECTOR:0.d,COUNTER_3:0.d,DXP_ICR:0.d,DXP_OCR:0.d,DXP_RAW:0.d,DXP_LIVE:0.d,$
    DXP1:0.d,ICR_DXP1:0.d,OCR_DXP1:0.d,LIVE_DXP1:0.d,FWHM_DXP1:0.d,DXP2:0.d,ICR_DXP2:0.d,OCR_DXP2:0.d,LIVE_DXP2:0.d,FWHM_DXP2:0.d,DXP3:0.d,ICR_DXP3:0.d,OCR_DXP3:0.d,$
    LIVE_DXP3:0.d,FWHM_DXP3:0.d,DXP4:0.d,ICR_DXP4:0.d,OCR_DXP4:0.d,LIVE_DXP4:0.d,FWHM_DXP:0.d,DXP5:0.d,ICR_DXP5:0.d,OCR_DXP5:0.d,LIVE_DXP5:0.d,FWHM_DXP5:0.d,$
    DXP6:0.d,ICR_DXP6:0.d,OCR_DXP6:0.d,LIVE_DXP6:0.d,FWHM_DXP6:0.d,DXP7:0.d,ICR_DXP7:0.d,OCR_DXP7:0.d,LIVE_DXP7:0.d,FWHM_DXP7:0.d,DXP8:0.d,ICR_DXP8:0.d,OCR_DXP8:0.d,$
    LIVE_DXP8:0.d,FWHM_DXP8:0.d,DXP9:0.d,ICR_DXP9:0.d,OCR_DXP9:0.d,LIVE_DXP9:0.d,FWHM_DXP9:0.d,DXP10:0.d,ICR_DXP10:0.d,OCR_DXP10:0.d,LIVE_DXP10:0.d,FWHM_DXP10:0.d,$
    DXP11:0.d,ICR_DXP11:0.d,OCR_DXP11:0.d,LIVE_DXP11:0.d,FWHM_DXP11:0.d,DXP12:0.d,ICR_DXP12:0.d,OCR_DXP12:0.d,LIVE_DXP12:0.d,FWHM_DXP12:0.d,DXP13:0.d,ICR_DXP13:0.d,$
    OCR_DXP13:0.d,LIVE_DXP13:0.d,FWHM_DXP13:0.d,DXP14:0.d,ICR_DXP14:0.d,OCR_DXP14:0.d,LIVE_DXP14:0.d,FWHM_DXP14:0.d,DXP15:0.d,ICR_DXP15:0.d,OCR_DXP15:0.d,LIVE_DXP15:0.d,$
    FWHM_DXP15:0.d,DXP16:0.d,ICR_DXP16:0.d,OCR_DXP16:0.d,LIVE_DXP16:0.d,FWHM_DXP16:0.d,DXP17:0.d,ICR_DXP17:0.d,OCR_DXP17:0.d,LIVE_DXP17:0.d,FWHM_DXP17:0.d,DXP18:0.d,$
    ICR_DXP18:0.d,OCR_DXP18:0.d,LIVE_DXP18:0.d,FWHM_DXP18:0.d,DXP19:0.d,ICR_DXP19:0.d,OCR_DXP19:0.d,LIVE_DXP19:0.d,FWHM_DXP19:0.d,DXP20:0.d,ICR_DXP20:0.d,OCR_DXP20:0.d,$
    LIVE_DXP20:0.d,FWHM_DXP20:0.d,DXP21:0.d,ICR_DXP21:0.d,OCR_DXP21:0.d,LIVE_DXP21:0.d,FWHM_DXP21:0.d,DXP22:0.d,ICR_DXP22:0.d,OCR_DXP22:0.d,LIVE_DXP22:0.d,FWHM_DXP22:0.d,$
    DXP23:0.d,ICR_DXP23:0.d,OCR_DXP23:0.d,LIVE_DXP23:0.d,FWHM_DXP23:0.d,DXP24:0.d,ICR_DXP24:0.d,OCR_DXP24:0.d,LIVE_DXP24:0.d,FWHM_DXP24:0.d,DXP25:0.d,ICR_DXP25:0.d,$
    OCR_DXP25:0.d,LIVE_DXP25:0.d,FWHM_DXP25:0.d,DXP26:0.d,ICR_DXP26:0.d,OCR_DXP26:0.d,LIVE_DXP26:0.d,FWHM_DXP26:0.d,DXP27:0.d,ICR_DXP27:0.d,OCR_DXP27:0.d,LIVE_DXP27:0.d,$
    FWHM_DXP27:0.d,DXP28:0.d,ICR_DXP28:0.d,OCR_DXP28:0.d,LIVE_DXP28:0.d,FWHM_DXP28:0.d,DXP29:0.d,ICR_DXP29:0.d,OCR_DXP29:0.d,LIVE_DXP29:0.d,FWHM_DXP29:0.d,DXP30:0.d,$
    ICR_DXP30:0.d,OCR_DXP30:0.d,LIVE_DXP30:0.d,FWHM_DXP30:0.d,DXP31:0.d,ICR_DXP31:0.d,OCR_DXP31:0.d,LIVE_DXP31:0.d,FWHM_DXP31:0.d,DXP32:0.d,ICR_DXP32:0.d,OCR_DXP32:0.d,$
    LIVE_DXP32:0.d,FWHM_DXP32:0.d,DXP33:0.d,ICR_DXP33:0.d,OCR_DXP33:0.d,LIVE_DXP33:0.d,FWHM_DXP33:0.d,DXP34:0.d,ICR_DXP34:0.d,OCR_DXP34:0.d,LIVE_DXP34:0.d,FWHM_DXP34:0.d,$
    DXP35:0.d,ICR_DXP35:0.d,OCR_DXP35:0.d,LIVE_DXP35:0.d,FWHM_DXP35:0.d,MONITOR:0.d,DXP_TOTAL:0.d}

 ; Renaming stc as ipro, for convenience.
  ipro = replicate(stc, file_lines(f1) - 1)

  readf, unit1, ipro
  close, unit1
  free_lun, unit1

  ; normalisation and dead-time calculations here.
    for i = 0, 34 do begin                           ; looping through pixels
      for ii = 0, n_elements(ipro.energy)-1 do begin ; looping through energy values
    
        ipro[ii].(i*5+12)  = (ipro[ii].(i*5+12) / ipro[ii].monitor)  * (ipro[ii].(i*5+13) / ipro[ii].(i*5+14))
       
      endfor
    endfor

  ; truncating for desired 'scan'
  ipro = ipro[0:580]             ; choosing 'first' repeat.
  ;ipro = ipro[581:581+580]      ; choosing 'second' repeat.
  ;ipro = ipro[1162:1162+580]    ; choosing 'third' repeat.
  ;ipro = ipro[1743+54:1743+634] ; choosing 'fourth' repeat.

  size_data = n_elements(ipro.energy)
  
  ; Read in i-pr HPA transmission data - as published in related 'transmission' XAFS paper (Chantler, 2015)
  f1 = path + 'iprHPAtrans15mMformatted.txt'

  header = ''

  openr, unit1, f1, /get_lun
  readf, unit1, header

  ; Defining structure of data columns
  stc =  {Energy:0.d, Energy_uncertainty:0.d, mu_rho:0.d}

  iproHPA = replicate(stc, file_lines(f1) - 1)

  readf, unit1, iproHPA
  close, unit1
  free_lun, unit1  
  
  loadct, 40                     ; loads colour table '40'

  ; Reading in FFAST data
  Mu_Ni = ffast_attenuation(2801, ipro.energy)     ; FFAST Nickel
  
  ; Lining up theoretical and experimental photoabsorption 'edge' (i.e., energy translation)
  maxfe  = max(Mu_Ni, maxf)                    ; max FFAST attenuation 
  maxdat = max(ipro.dxp15, maxd)               ; experimental fluorescence maximum 
  maxgap = ipro[maxf].energy-ipro[maxd].energy ; energy difference between theoretical and experimental maxima i.e., magnitude of required energy translation.

  ipro.energy = ipro.energy+maxgap ; Slight energy calibration, so that the top of the 'edge' for theory and experiment line up.

  ; [mu / rho] data from FFAST for elements as listed in variable name.
  Mu_Helium   = ffast_attenuation(2,    ipro.energy)  
  Mu_Kapton   = ffast_attenuation(9999, ipro.energy)  
  Mu_Silicone = ffast_attenuation(8888, ipro.energy)  
  Mu_Nitrogen = ffast_attenuation(7,    ipro.energy) 
  Mu_Oxygen   = ffast_attenuation(8,    ipro.energy) 
  Mu_Argon    = ffast_attenuation(18,   ipro.energy) 
  Mu_ACN      = ffast_attenuation(7777, ipro.energy)  ; acetonitrile (solvent)
  Mu_BCN      = ffast_attenuation(7778, ipro.energy)  ; butyronitrile (solvent)
  F2_Ni       = ffast_attenuation(2800, ipro.energy)  ; imaginary component of atomic form factor from FFAST.

  Mu_Air      = 0.78 * Mu_Nitrogen + 0.21 * Mu_Oxygen + 0.093 * Mu_Argon
  Mu_solvent  = (Mu_ACN * 0.4 + Mu_BCN * 0.6) * 0.5515 ; Representing ratios of solvent components, multiplied by [rho * t] as quoted in Chantler (JSR, 2015)

  ; read in full array of FFAST Ni data
  f2 = path + 'ffast_files/ctc_niff_ryan.dat'

  header = ''

  openr, unit2, f2, /get_lun
  readf, unit2, header

  ; Defining structure of data columns
  stc1 =  {Energy:0.d, f1:0.d, f2:0.d, pe:0.d, coh:0.d, tot:0.d, k:0.d, lambda:0.d}

  NiFFAST = replicate(stc1, file_lines(f2) - 1)

  readf, unit2, NiFFAST
  close, unit2
  free_lun, unit2

  NiFFAST_tot = ffast_attenuation_ryan(ipro.energy,1) ; FFAST Nickel total attenuation
  NiFFAST_k   = ffast_attenuation_ryan(ipro.energy,2) ; FFAST Nickel attentuation due to K-edge  
 

  ; Accounting for mass fraction on Nickel in i-pr/n-pr molecule (same molecular weight for each molecule).
  ; Molecular weight of molecule: 383.1100 amu
  ; Atomic mass of Nickel: 58.6934 amu
  ; Therefore, relevant FFAST tabulated data should be scaled to reflect this.

  NiFFAST_tot = NiFFAST_tot * (58.6934/383.1100)
  NiFFAST_k   = NiFFAST_k   * (58.6934/383.1100) 

  ; Zeroing all the NaNs at the start of array
  ; Similarly, zeroing the artificial 'slope' on the edge of niffast_tot. This is a side-effect of interpolation process.
  for ee = 0,140 do begin

    niffast_k[ee]   = 0d
    niffast_tot[ee] = 0d

  endfor
  ; ---------- Fitting 1D model ------------------------
  ; The amplitude of each spectra must be scaled somehow. A simple option is to scale so that they are the same at the peak of the edge.
  ; Choose one of the central pixels, and define this as the 'reference' pixel.
  y_ref = ipro.dxp15
  
  h_val = 1.d             ; toggle horizontal angle fitting.
  v_val = 1.d             ; vertical angle fitting.
  P     = dindgen(85) * 0 ; initialising parameter array for MPFIT

  ; Initial guess parameters  
  ; P[0:35] = PIXEL AMPLITUDE
  P[0:35] = 1d
  p[0]  = 0.d    ; pixel is defective. 
  p[31] = 0.d    ; pixel is only about half as sensitive as the rest, and should be excluded.
  p[33] = 0.d    ; pixel is defective.
  p[35] = 0.d    ; Pixel 36 is broken and permanently disabled by the Synchrotron.
  
  ; MODEL PARAMETERS
  p[38] = 0.d     ; offset from 45deg
  p[39] = 0.082d  ; (meters) L parameter - distance from cryostat to Ge detector

  p[41] = 42.8    ; mu_f / rho in (cm^2/g)
  p[42] = 1d      ; A
  p[43] = 0.5515d ; From 2015 paper

  ; Fixing and limiting parameters
  parinfo = replicate({fixed:0, limited:[0,0], limits:[0.,0.], tied:''}, n_elements(P)) 
  
  parinfo[43].fixed = 1.d ; fixing to value defined in publication
  parinfo[39].fixed = 1.d 
  parinfo[41].fixed = 1.d ; muf/rho taken from FFast at Ni K_alpha energy.
  parinfo[42].fixed = 1.d

  parinfo[15].fixed = 1.d ; y_ref pixel
  parinfo[0].fixed  = 1.d ; Dead pixels
  parinfo[31].fixed = 1.d ; " "
  parinfo[33].fixed = 1.d ; " "
  parinfo[35].fixed = 1.d ; " "

  parinfo[38].limited   = 1.d
  parinfo[38].limits[0] = -0.0872665d & parinfo[38].limits[1] = 0.0872665d ; allowing for +/- 5 degree drift

  print, 'Now fitting horizontal and vertical angles...'
  FUNCTARG1 = {sample:ipro, raw:ipro, y_ref:y_ref, Mu_Ni:Mu_Ni, f2_ni:f2_ni, h_val:h_val, v_val:v_val, Mu_Air:mu_air, Mu_Kapton:mu_kapton, Mu_Helium:mu_helium, $
               Mu_Silicone:Mu_silicone, niffast_tot:niffast_tot, niffast_k:niffast_k}
  P = MPFIT('beamline_model',P,FUNCTARGS=FUNCTARG1, parinfo=parinfo, quiet=1)
  print, 'Horizontal and vertical angles fitted.'
  
  p_vals = p ; Storing fitted parameters in 'p_vals' array
  
  ; Scaling fluorescence data to match amplitude of transmission data.
  transmission_scale = dindgen(35)*0
  max_hpd_energy = max(ipro.energy, i)
  edge = where(ipro.energy ge 8.352+maxgap)
  max_hpd_energy_location = n_elements(ipro.energy)-1
  Energy_range = where(iprohpa.energy ge 8352)
  transmission_spectra_amplitude = iprohpa[energy_range[0]].mu_rho * 0.0018377 - 0.020 ; transmission [mu / rho] * [rho t] minus slight background offset
  
  ; reversing effect of previous energy translation
  ipro.energy = ipro.energy - maxgap
    
  air_ = 0 ; Toggling incident angle argument for theta calculations (whether incident angle is 90 deg or not)
  thetaout = theta_calculation_v1(P_vals, h_val, v_val, air_) ; Getting theta parameters.
  deviates = dindgen(n_elements(ipro.energy))*0d              ; Creating array that will be used in fitting optimisation
  intensity_scale = dindgen(n_elements(ipro.energy))*0d       ; Creating array that will store result of fitting.
  
  ; Dealing with "efficiency" term - eq 5 in 2012 Chantler paper
  quantum_efficiency = 1.d ; This should be a function of energy (perhaps different for each pixel), difficult to implement rigorously.
  ; Equation (6) in paper: theta_w_horiz ~= theta_air_horiz ~= (theta_out_horiz - 45deg.)
  air_ = 1
  thetaout_air = theta_calculation_v1(P_vals, h_val, v_val, air_)
  for ooo = 0, 35 do begin
    if (ooo eq 0) or (ooo eq 31) or (ooo eq 33) or (ooo eq 35) then goto, skip1000 ; excluding broken pixels.
    intensity_scale     = beamline_variables(ooo, ipro, Mu_Air, Mu_Kapton, Mu_Helium, Mu_Silicone, thetaout_air)
        
    transmission_scale[ooo] = transmission_spectra_amplitude/(ipro[edge[0]].(ooo*5+12)*p_vals[ooo]*intensity_scale[edge[0]])
     
    ipro.(ooo*5+12) = ipro.(ooo*5+12)*p_vals[ooo]*intensity_scale*transmission_scale[ooo] ; updating data array
    
    skip1000:
  endfor
  
 ; Scaling FFAST data to reflect concentration of i-pr/n-pr complex in dilute sample.
  FFAST_factor = ipro[i].dxp15 / niffast_tot[i]
  niffast_tot  = niffast_tot * FFAST_factor
  niffast_k    = niffast_k * FFAST_factor

  ; Fixing and limiting parameters
  parinfo = replicate(  { fixed:0, limited:[0,0], limits:[0.,0.], tied:''} ,n_elements(P)) 
  
  parinfo[15].fixed = 1d 
  parinfo[39].fixed = 1d
  parinfo[41].fixed = 1d
  parinfo[42].fixed = 1d
  
  parinfo[38].limited   = 1.d
  parinfo[38].limits[0] = -0.0872665d & parinfo[38].limits[1] = 0.0872665d ; allowing for +/- 5 degree drift

  print, 'Now fitting horizontal and vertical angles...'
  Fitted_p4 = dindgen(85)
  FUNCTARG1 = { sample:ipro, raw:ipro, y_ref:y_ref, Mu_Ni:Mu_Ni, f2_ni:f2_ni, h_val:h_val, v_val:v_val, Mu_Air:mu_air, Mu_Kapton:mu_kapton, Mu_Helium:mu_helium, Mu_Silicone:Mu_silicone} 
  Fitted_P4 = MPFIT('linear_seafflux_model',P,COVAR=covar1,FUNCTARGS=FUNCTARG1, parinfo=parinfo, PERROR=perror,autoderivative=1,BESTNORM=chisq,quiet=1, status = status, ftol = 1d-20, xtol = 1d-20, gtol = 1d-20, dof=dof)
  print, 'Horizontal and vertical angles fitted.'  
  
    ; Updating data array with result of fitting routine.
    pre_index = where(ipro.energy le 8.375) ; Pre-'edge' energy range
    index = where(ipro.energy gt 8.375)     ; Post-'edge' energy range
    xin = ipro.energy
    air_ = 0
    thetaout = theta_calculation_v1(fitted_p4, h_val, v_val, air_) ; getting theta parameters.
    deviates = xin * 0.d
    intensity_scale = xin * 0.d
    ; dealing with "efficiency" term - eq 5 in 2012 Chantler paper
    quantum_efficiency = 1.d ; This should be a function of energy, difficult to implement rigorously.
    PathLengthFactor   = dindgen(35)*0
    ; Equation (6) in 2015 paper: theta_w_horiz ~= theta_air_horiz ~= (theta_out_horiz - 45deg.)
    air_ = 1
    thetaout_air = theta_calculation_v1(fitted_p4, h_val, v_val, air_)
    
    ; Constants
    Planck = double(6.626070040e-34) ;(J/s)
    v_light = 299792458d ;(m/s)
    r_electron = 2.8179403267e-15 ;(m)
    u_mass = 1.660539040e-27 ; (kg)
    Relative_atomic_mass_Ni = 58.6934d

    f2_factor = (2 * Planck * v_light * r_electron) / (ipro[index].energy * 1.60218e-16 * u_mass * Relative_atomic_mass_Ni)  ; From Martin de Jonge's thesis, pg 26.
        
    for ooo = 0, 35 do begin
      if (ooo eq 0) or (ooo eq 31) or (ooo eq 33) or (ooo eq 35) then goto, skip1003 ; excluding dead or dodgy pixels.
      
      intensity_scale = (fitted_P4[ooo] * f2_factor * f2_Ni[index] * cos(!pi/4) /  ((Mu_Ni[index])/cos(!pi/4) + (fitted_P4[41]/cos(thetaout[ooo])) )) * $
                        (fitted_P4[42] - exp( - (fitted_P4[43] * Mu_Ni[index])/cos(!pi/4) - fitted_P4[41]*fitted_P4[43]/cos(thetaout[ooo]) ) )
        
      ipro[index].(ooo*5+12) = ipro[index].(ooo*5+12)*intensity_scale
  
      skip1003:
    endfor  
  
    ; fitting to scale amplitude to transmission data

    p = dindgen(35)*0+1

    ; Fixing and limiting parameters
    parinfo = replicate({fixed:0, limited:[0,0], limits:[0.,0.], tied:''}, n_elements(P))

    parinfo[14].fixed = 1.d

    FUNCTARG7 = {ipro:ipro, hpa:iprohpa} 
    P = MPFIT('transmission_scale_fit',P,FUNCTARGS=FUNCTARG7, parinfo=parinfo, quiet=1)

    trans_p = p

    transmission_scale = dindgen(35)*0
    max_hpd_energy = max(ipro.energy, i)
    max_hpd_energy_location = n_elements(ipro.energy)-1 
    edge = where(ipro.energy ge 8.600)
    Energy_range = where(iprohpa.energy ge 8600);where(iprohpa.energy ge max_hpd_energy*1000)
    transmission_spectra_amplitude = iprohpa[energy_range[0]].mu_rho * 0.0018377 - 0.02
    
    region = where(ipro.energy ge 8.375)

    for qqq = 0, 35 do begin
      if (qqq eq 0) or (qqq eq 31) or (qqq eq 33) or (qqq eq 35) then goto, skip1017 ; excluding broken pixels.

      transmission_scale[qqq] = transmission_spectra_amplitude/(ipro[edge[0]].(qqq*5+12)*trans_p[qqq-1])

      ipro[region].(qqq*5+12) = ipro[region].(qqq*5+12)*trans_p[qqq-1]*transmission_scale[qqq] 

      skip1017:
    endfor
    
  thetaout = theta_calculation_v1(P_vals, h_val, v_val, air_) ; getting theta parameters.
  
  P     = dindgen(85) * 0 + 1 ; initialising parameter array for MPFIT

  ; Fixing and limiting parameters
  parinfo = replicate({fixed:0, limited:[0,0], limits:[0.,0.], tied:''}, n_elements(P))
  parinfo[14].fixed = 1d

  print, 'Now fitting chi...'
  FUNCTARG1 = {sample:ipro, raw:ipro, niffast_tot:niffast_tot, niffast_k:niffast_k, mu_solvent:mu_solvent} ; needed for MPFIT
  P = MPFIT('self_absorption_chi_model',P,FUNCTARGS=FUNCTARG1, parinfo=parinfo, quiet=1)
  print, 'Chi fitted.'
  
  print, 'Now plotting chi-corrected spectra..."
  
  air_ = 0
  thetaout = theta_calculation_v1(P, h_val, v_val, air_) - 90*!pi/180.; getting theta parameters.
  deviates = xin * 0.d
  mu_f_on_rho = 42.8
  
  p_chi = p
  
  region = where(ipro.energy ge 8.375)
  
  for mmm = 0, 35 do begin

    if (mmm eq 0) or (mmm eq 31) or (mmm eq 33) or (mmm eq 35) then goto, skip1212 ; Excluding broken pixels.

      ; Variables names selected to match theory outlined in Appendix of IUCrJ publication
      a = ipro[region].(mmm*5+12)
      b = niffast_k[region]
      c = (niffast_tot[region] - niffast_k[region]) + Mu_solvent[region] + (mu_f_on_rho)/cos(thetaout[mmm])     ; [mu / rho]_other

      fitted_chi = chi_inversion_cubic_v2(a,b,c)

      c_exp = (niffast_tot[region] - niffast_k[region]) + Mu_solvent[region]
      d = 0.0018377d                                                             ; rho * t - from JSR 2015 publication - Table 5 in units g/cm^2
      e = mu_f_on_rho                                                            ; muf/rho in cm^2/g
      f = cos(thetaout[mmm])                                                     ; theta_out of 15th pixel

       
    ; Updating ipro array here    
    ipro[region].(mmm*5+12) = p_chi[mmm]*fitted_chi + niffast_k[region]

    skip1212:

  endfor
  
; fitting to fix amplitude to transmission results
  p = dindgen(35)*0+1

  ; Fixing and limiting parameters
  parinfo = replicate({fixed:0, limited:[0,0], limits:[0.,0.], tied:''}, n_elements(P))
 
  parinfo[14].fixed = 1.d 

  FUNCTARG7 = {ipro:ipro, hpa:iprohpa} 
  P = MPFIT('transmission_scale_fit',P,FUNCTARGS=FUNCTARG7, parinfo=parinfo, quiet=1)
  
  trans_p = p
  
  transmission_scale = dindgen(35)*0
  max_hpd_energy = max(ipro.energy, i)
  max_hpd_energy_location = n_elements(ipro.energy)-1
  Energy_range = where(iprohpa.energy ge max_hpd_energy*1000)
  transmission_spectra_amplitude = iprohpa[energy_range[0]].mu_rho * 0.0018377
  
  edge = where(ipro.energy ge 8.400)
  Energy_range = where(iprohpa.energy ge 8400)
  transmission_spectra_amplitude = iprohpa[energy_range[0]].mu_rho * 0.0018377 - 0.0225

  region = where(ipro.energy ge 8.375)
  
  p = plot(dindgen(5)*0, dindgen(5)*0, xrange=[8.2,9.4], yrange=[0,0.135], thick=3, xtitle = 'Energy (keV)', ytitle = 'PLACEHOLDER', title='Chi-corrected fit')

  for qqq = 0, 35 do begin
    if (qqq eq 0) or (qqq eq 31) or (qqq eq 33) or (qqq eq 35) then goto, skip1010 ; Excluding broken pixels.

    transmission_scale[qqq] = transmission_spectra_amplitude/(ipro[edge[0]].(qqq*5+12)*trans_p[qqq-1])

    ipro[region].(qqq*5+12) = ipro[region].(qqq*5+12)*trans_p[qqq-1]*transmission_scale[qqq] ; updating data array
    
   p = plot(ipro.energy, ipro.(qqq*5+12), color=cgcolor(qqq*7), /OVERPLOT)

    skip1010:
  endfor
  
END

FUNCTION self_absorption_chi_model, P, sample=sample, niffast_tot=niffast_tot, niffast_k=niffast_k, raw=raw, mu_solvent=mu_solvent
; Note - this should only be fitting in region above edge, since chi is zero below edge.

  region = where(sample.energy ge 8.35) ; Region above edge

  air_ = 0
  h_val = 1
  v_val = 1
  thetaout = theta_calculation_v1(P, h_val, v_val, air_) - 90*!pi/180; getting theta parameters.
  deviates = dindgen(n_elements(raw.energy)) * 0.d
  mu_f_on_rho = 42.8 ; (cm^2/g)
  
  for sss = 0, 35 do begin

    if (sss eq 0) or (sss eq 31) or (sss eq 33) or (sss eq 35) then goto, skip ; Excluding broken pixels.
    
    a = sample[region].(sss*5+12)
    b = niffast_k[region]
    c =  Mu_solvent[region] + (mu_f_on_rho)/cos(thetaout[sss])     ; [mu / rho]_other
    c_ref =  Mu_solvent[region] + (mu_f_on_rho)/cos(thetaout[14]) ; [mu / rho]_other_ref
    
;######################################################################################################   
                                                                                                     ;#
   ; Choose chi_inversion_v2() for quadratic expansion or chi_inversion_cubic() for cubic expansion. ;#
   ; Further mathematical details as described in Appendix of IUCrJ publication.                     ;#
    fitted_chi     = chi_inversion_cubic_v2(a,b,c)                                                   ;#
    fitted_chi_ref = chi_inversion_cubic_v2(sample[region].dxp15,b,c_ref)                            ;#
                                                                                                     ;#
;######################################################################################################

    deviates[sss] = total(p[sss]*fitted_chi - p[15]*fitted_chi_ref)
    skip:

  endfor

  RETURN, deviates
END

FUNCTION beamline_model, P, sample=sample, y_ref = y_ref, raw=raw, Mu_Ni = Mu_Ni, f2_ni = f2_ni,  h_val=h_val, v_val=v_val, Mu_Air=mu_air, $
                          Mu_Kapton=mu_kapton, Mu_Helium=mu_helium, Mu_Silicone=mu_silicone, niffast_tot=niffast_tot, niffast_k=niffast_k

air_ = 0
thetaout = theta_calculation_v1(P, h_val, v_val, air_) ; getting theta parameters.
deviates = dindgen(n_elements(sample.energy))*0d
intensity_scale = dindgen(n_elements(sample.energy))*0d

; Equation (6) in paper: theta_w_horiz ~= theta_air_horiz ~= (theta_out_horiz - 45deg.)
air_ = 1
thetaout_air = theta_calculation_v1(P, h_val, v_val, air_)
for sss = 0, 35 do begin

  if (sss eq 0) or (sss eq 31) or (sss eq 33) or (sss eq 35) then goto, skip ; excluding dead or dodgy pixels.

  intensity_scale     = beamline_variables(sss, raw, Mu_Air, Mu_Kapton, Mu_Helium, Mu_Silicone, thetaout_air)
  intensity_scale_ref = beamline_variables(15,  raw, Mu_Air, Mu_Kapton, Mu_Helium, Mu_Silicone, thetaout_air)
  
  deviates[sss] = total( P[sss] * sample.(sss*5+12) * intensity_scale - P[15] *  sample.dxp15 * intensity_scale_ref)
  skip:

endfor

RETURN, deviates
END

FUNCTION beamline_variables, sss, raw, Mu_Air, Mu_Kapton, Mu_Helium, Mu_Silicone, thetaout_air, quantum_efficiency

  ; dealing with "efficiency" term - eq 5 in 2012 Chantler paper
  quantum_efficiency = 1.d ; This should be a function of energy (perhaps different for each pixel), difficult to implement rigorously.
  
  PathLengthFactor   = dindgen(35)*0

  muf_on_rho_air    = 13.2989d ; Units: (cm^2)/g - Get air mixture FFAST data from Martin's IDL program, then pick Nickel K_alpha energy for fluorescent photon energy.
  rho_t_air         = 0.0011 *  37.4 ;(g/cm^3) * cm - taken from Chantler (JSR 2015)
  theta_air         = thetaout_air[sss]

  muf_on_rho_kapton = 8.994418d  ; Units: (cm^2)/g
  rho_t_kapton      = 1.42 * 0.0160 ; g/cm^3 * cm  7/8 accounts for one less kapton window compared to absorption measurement.
  theta_kapton      = thetaout_air[sss]

  muf_on_rho_det_gas= 8.88848d  
  rho_t_det_gas     = 0.0012 * 19.0 ; (g/cm^3) * cm
  theta_det         = thetaout_air[sss]

  muf_on_rho_helium = 0.290224 
  rho_t_helium      = 0.0001785 * 2.2 * (1d/2d) ; (g/cm^3) * cm
  theta_helium      = thetaout_air[sss]

  muf_on_rho_silicone = 31.7594 
  rho_t_silicone    = 0.968 * 0.0088 ; (g/cm^3) * cm
  theta_silicone    = thetaout_air[sss]

  ;We should multiply the detected intensity through by 1/PathLengthFactor to retrieve the (pathlength attenuation)-corrected spectra.
  PathlengthFactor[sss] = quantum_efficiency * exp( -(muf_on_rho_air * rho_t_air)/cos(theta_air)        - (3d/4d)*(muf_on_rho_kapton * rho_t_kapton)/cos(theta_kapton    ) $
    - (1d/4d)*(muf_on_rho_kapton * rho_t_kapton)/cos(theta_kapton     + !pi/4) - (muf_on_rho_silicone * rho_t_silicone)/cos(theta_silicone   + !pi/4)  $
    - (muf_on_rho_det_gas * rho_t_det_gas)/cos(theta_det       )  -(muf_on_rho_helium * rho_t_helium)/cos(theta_helium       )          )   ; overall airpath/window correction factor i.e., eq (5) in JSR 2012 paper. Ratios are relative to absorption setup values quoted in paper.

  PathlengthFactor[15]  = quantum_efficiency * exp( -(muf_on_rho_air * rho_t_air)/cos(thetaout_air[15]) - (3d/4d)*(muf_on_rho_kapton * rho_t_kapton)/cos(thetaout_air[15]) $
    - (1d/4d)*(muf_on_rho_kapton * rho_t_kapton)/cos(thetaout_air[15] + !pi/4) - (muf_on_rho_silicone * rho_t_silicone)/cos(thetaout_air[15] + !pi/4)  $
    - (muf_on_rho_det_gas * rho_t_det_gas)/cos(thetaout_air[15])  -(muf_on_rho_helium * rho_t_helium)/cos(thetaout_air[15]   )          )   ; overall airpath/window correction factor i.e., eq (5) in JSR 2012 paper.

  ETerm = dindgen(n_elements(raw.energy))*0

  ETerm = path_length_attenuation_E_v1(raw, Mu_Air, Mu_Kapton, Mu_Helium, Mu_Silicone)

  ; Term accounting for attenuation due to beamline apparatus.
  intensity_scale     = 1d/(Eterm * PathLengthFactor[sss])
  
  return, intensity_scale
END

Function path_length_attenuation_E_v1, ipro, Mu_Air, Mu_Kapton, Mu_Helium, Mu_Silicone  ; Energy-dependent correction for incoming photons from synchrotron source.

  rho_t_air         = 0.0011 * (37.4 - 8.1) ; (g/cm^3) * cm - taken from Chantler (JSR 2015)
  theta_air         = 0.d

  ; NOTE: THIS IS FOR ONE WINDOW - T MUST BE MULTIPLIED BY NUMBER OF WINDOWS
  rho_t_kapton      = 1.42 * (1.60 * (5/8)) * (1/100) ; g/cm^3 * cm  7/8 accounts for one less kapton window than in absorption measurement.
  ; 3 Kapton windows are perp. , one is at 45 deg.

  rho_t_helium      = 0.0001785 * 2.2 * (1d/2d)       ; only half the cryostat gas gets exposed to all energies.
  theta_helium      = 0.d

  rho_t_silicone    = 0.968 * 0.0088  * (5d/8d)       ; only half the silicone gets exposed to all energies.
  theta_silicone    = !pi/4d

  PathlengthFactor = exp( - ((Mu_Air * rho_t_air)/cos(theta_air)) - ((Mu_Kapton * 3 * rho_t_air)/cos(0)) - ((Mu_Kapton * rho_t_Kapton)/cos(!pi/4d)) - ((Mu_Helium * rho_t_helium)/cos(theta_helium)) - $
                     ((Mu_Silicone * rho_t_silicone)/cos(theta_silicone)) )   ; incoming airpath/window correction factor i.e., adjusted version of eq (5) in JSR 2012 paper.

  return, PathlengthFactor
end

FUNCTION channel_separation_v1

  channel_sep = reverse(float(indgen(36)/6-2.5))

  RETURN,  channel_sep
END

FUNCTION transmission_scale_fit, ipro=ipro, hpa=hpa, P

edge = where(ipro.energy ge 8.35)
deviates = ipro.energy*0

  for rrr = 1, 34 do begin
    
    if (rrr eq 31) or (rrr eq 33) then goto, skip90    
    
    deviates[rrr] = total(p[rrr] * ipro[edge].(rrr*5+12) - p[14]*ipro[edge].dxp15)
    
    skip90:
  endfor
  
return, deviates
END

FUNCTION linear_seafflux_model, P, sample=sample, y_ref = y_ref, ndets=ndets, raw=raw, Mu_Ni = Mu_Ni, f2_ni = f2_ni,  scale = scale, start = start, count = count, h_val=h_val, v_val=v_val, Mu_Air=mu_air, Mu_Kapton=mu_kapton, Mu_Helium=mu_helium, Mu_Silicone=mu_silicone

  index = where(raw.energy gt 8.375)

  thetaout = theta_calculation_v2(P, h_val, v_val) ; getting theta parameters.
  deviates = dindgen(n_elements(sample.energy))*0d
  intensity_scale = dindgen(n_elements(sample.energy))*0d

  for sss = 0, 35 do begin

    if (sss eq 0) or (sss eq 31) or (sss eq 33) or (sss eq 35) then goto, skip ; Excluding broken pixels.
    
    ; Constants
    Planck = double(6.626070040e-34) ;(J/s)
    v_light = 299792458d ;(m/s)
    r_electron = 2.8179403267e-15 ;(m)
    u_mass = 1.660539040e-27 ; (kg)
    Relative_atomic_mass_Ni = 58.6934d
 
    ; From Martin de Jonge's thesis, pg 26
    f2_factor = (2 * Planck * v_light * r_electron) / (raw[index].energy * 1.60218e-16 * u_mass * Relative_atomic_mass_Ni)  

    intensity_scale =  (P[sss] * f2_factor * f2_Ni[index] * cos(!pi/4) /  ((Mu_Ni[index])/cos(!pi/4) + (P[41]/cos(thetaout[sss])) )) * $
                       ( P[42] - exp( - (P[43] * Mu_Ni[index])/cos(!pi/4) - P[41]*P[43]/cos(thetaout[sss]) ) )

    ; f = fluorescent yield ~ const.
    ; I_0 = incident X-ray beam intensity ~ const. (but energy dependent?)
    ; Omega = solid angle ~ const. to first order
    ; (mu/rho) is given by FFast data
    ; Theta_inc also seems to be a const. (45deg) to first order.
    ; Theta_out is obviously a variable, and a function of theta_v and theta_h
    ; Overall scaling P[sss] - probably need one of these for each pixel, for now.
    ; muf/rho = P[41]
    ; A = P[42]
    ; rho*t = P[43]

    ; self-absorption correction for 15th pixel ~ reference.
    intensity_scale_ref =  (P[15] * f2_factor * f2_Ni[index] * cos(!pi/4) /  ((Mu_Ni[index])/cos(!pi/4) + (P[41]/cos(thetaout[15])) )) * $
                           ( P[42] - exp( - (P[43] * Mu_Ni[index])/cos(!pi/4) - P[41]*P[43]/cos(thetaout[15]) ) )

    deviates[sss] = total((intensity_scale) * sample[index].(sss*5 + 12) - (intensity_scale_ref)*y_ref[index])
    skip:

  endfor

  RETURN, deviates

END

FUNCTION theta_calculation_v1, p, h_val, v_val, air_

  channel_sep = -1 * channel_separation_v1() ; effectively n in paper equation

  channel_height = abs([2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5]) ; effectively m in paper equation

  ; Channel_sep equivalent to n, number of pixel elements from central point. We're considering centre of 6 x 6 pixel detector, and middle point of each pixel.

  ; 8.4mm is equivalent to C, separation between pixel centres.

  if h_val eq 0 then thetah = dindgen(36)*0
  if h_val eq 1 then thetah = !pi/4+p[38]+( atan(channel_sep*0.0084/p[39]) )  ; paper: theta_out_h = 45deg + atan(n*C/L) where n is number of channel elements to central point.

  if air_ eq 1 then thetah = p[38]+( atan(channel_sep*0.0084/p[39]) )

  if v_val eq 0 then thetav = dindgen(36)*0
  if v_val eq 1 then begin
    thetav = atan(channel_height*0.0084/p[39])       ; paper: theta_out_v = acos( atan( m * C / L)) where m is the number of channel elements from the plane of incidence in the vertical axis.
  endif

  thetaout  = acos( cos(thetah)*cos(thetav) )        ; paper: (same equation)

  RETURN, thetaout
END

FUNCTION theta_calculation_v2, p, h_val, v_val

  channel_sep = -1 * channel_separation_v1() ; effectively n in paper equation

  channel_height = abs([2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5,2.5,1.5,0.5,-0.5,-1.5,-2.5]) ; effectively m in paper equation

  ; 8.4mm is equivalent to C, separation between pixel centres.

  if h_val eq 0 then thetah = dindgen(36)*0
  if h_val eq 1 then thetah = !pi/4+p[38]+( atan(channel_sep*0.0084/p[39]) )  ; paper: theta_out_h = 45deg + atan(n*C/L) where n is number of channel elements to central point.

  if v_val eq 0 then thetav = dindgen(36)*0
  if v_val eq 1 then begin
    thetav = atan(channel_height*0.0084/p[39])       ; paper: theta_out_v = acos( atan( m * C / L)) where m is the number of channel elements from the plane of incidence in the vertical axis.
  endif

  thetaout  = acos( cos(thetah)*cos(thetav) )        ; paper: (same equation)

  RETURN, thetaout
END

function ffast_attenuation,z,e

  get_path = cgsourcedir()
  path = get_path + 'ffast_files/'

  filename =  strtrim(string(z),2)+'.csv'
  data     = read_csv2(path+filename)

  y        = alog(1.D*data.mu_rho)

  interpolates = interpol(y,alog(data.energy),alog(E))

  return,exp(interpolates)
end

function ffast_attenuation_ryan,e,choice

  path = cgsourcedir()

  filename =  '/Users/rtrevorah/IDLWorkspace/Ni_complexes/fluorescence_qxafs_R/ref_data/FFAST/ctc_niff_ryan.dat'

  header = ''

  openr, unit2, filename, /get_lun
  readf, unit2, header

  ; Defining structure of data columns
  stc1 =  {Energy:0.d, f1:0.d, f2:0.d, pe:0.d, coh:0.d, tot:0.d, k:0.d, lambda:0.d}

  NiFFAST = replicate(stc1, file_lines(filename) - 1)

  readf, unit2, NiFFAST
  close, unit2
  free_lun, unit2

  if choice eq 1 then y = alog(1.D*niffast.tot)

  if choice eq 2 then y = alog(1.D*niffast.k)

  interpolates = interpol(y,alog(niffast.energy),alog(E))

  return,exp(interpolates)
end

FUNCTION chi_inversion_v2, a,b,c

  value = c/2. - b/2. - (2*b*c - 4*a*c^2 + b^2 + c^2)^(1/2.)/2.

  return, value
END

FUNCTION chi_inversion_cubic_v2, a,b,c

  value = ((- c^2 + b*c)^3/(27*(b - c)^3) - (a*c^4 - b*c^3)/(2.*(b - c)) + (((- c^3 + b*c^2)/(3.*(b - c)) - (- c^2 + b*c)^2/(9.*(b - c)^2))^3 + ((a*c^4 - b*c^3)/(2.*(b - c)) - $
    (- c^2 + b*c)^3/(27.*(b - c)^3) + ((- c^3 + b*c^2)*(- c^2 + b*c))/(6.*(b - c)^2))^2)^(1./2.) - ((- c^3 + b*c^2)*(- c^2 + b*c))/(6.*(b - c)^2))^(1./3.) - $
    ((- c^3 + b*c^2)/(3*(b - c)) - (- c^2 + b*c)^2/(9.*(b - c)^2))/((b*c - c^2)^3/(27.*(b - c)^3) - (a*c^4 - b*c^3)/(2*b - 2*c) + (((a*c^4 - b*c^3)/(2*b - 2*c) - $
    (b*c - c^2)^3/(27.*(b - c)^3) + ((b*c^2 - c^3)*(b*c - c^2))/(6.*(b - c)^2))^2 - ((b*c - c^2)^2/(9.*(b - c)^2) - (b*c^2 - c^3)/(3*b - 3*c))^3)^(1./2.) - $
    ((b*c^2 - c^3)*(b*c - c^2))/(6.*(b - c)^2))^(1./3.) + (- c^2 + b*c)/(3*(b - c))

  return, value
END